/******************************************************************************/
/*                                                                            */
/*                    Port9240 Driver for Windows NT/2000/XP                  */
/*                        Version 2.0, 12th January 2002                      */
/*                          http://www.beyondlogic.org                        */
/* modifications by D. Talmage (c) 2005 Allen Datagraph Systems - All Right Reserved */
/* permission is given to Owners of Allen Datagraph 9240 equipment to use this       */
/* driver while talking to a 9240 recorder                                           */
/*                                                                            */
/* Copyright  2002 Craig Peacock. Craig.Peacock@beyondlogic.org              */
/* Any publication or distribution of this code in source form is prohibited  */
/* without prior written permission of the copyright holder. This source code */
/* is provided "as is", without any guarantee made as to its suitability or   */
/* fitness for any particular use. Permission is herby granted to modify or   */
/* enhance this sample code to produce a derivative program which may only be */
/* distributed in compiled object form only.                                  */
/******************************************************************************/

#include <ntddk.h>
#include <..\Port9240_IOCTL.h>

NTSTATUS Port9240DeviceControl(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp);
VOID Port9240Unload(IN PDRIVER_OBJECT DriverObject);

NTSTATUS Port9240CreateDispatch(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
    )
{
    Irp->IoStatus.Information = 0;
    Irp->IoStatus.Status = STATUS_SUCCESS;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);
    return STATUS_SUCCESS;
}

NTSTATUS DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
    )
{
    PDEVICE_OBJECT deviceObject;
    NTSTATUS status;
    WCHAR NameBuffer[] = L"\\Device\\Port9240";
    WCHAR DOSNameBuffer[] = L"\\DosDevices\\Port9240";
    UNICODE_STRING uniNameString, uniDOSString;

    KdPrint( ("Port9240: Port9240 V1.0 has Loaded") );

    RtlInitUnicodeString(&uniNameString, NameBuffer);
    RtlInitUnicodeString(&uniDOSString, DOSNameBuffer);

    status = IoCreateDevice(DriverObject,
                            0,
                            &uniNameString,
                            FILE_DEVICE_UNKNOWN,
                            0,
                            FALSE,
                            &deviceObject);

    if(!NT_SUCCESS(status))
        return status;

    status = IoCreateSymbolicLink (&uniDOSString, &uniNameString);

    if (!NT_SUCCESS(status))
        return status;

    DriverObject->MajorFunction[IRP_MJ_CREATE] = Port9240CreateDispatch;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = Port9240DeviceControl;
    DriverObject->DriverUnload = Port9240Unload;

    return STATUS_SUCCESS;
}

/* Reads 9240 status data and waits for not busy to occur */
int WaitNotBusy(int TimeOutCount, PUCHAR lptportc)
{
  int idx;
  for(idx = 0; idx < TimeOutCount; idx++)
    if(READ_PORT_UCHAR(lptportc) & 0x80)return(0); // normal return when busy goes away
  return(1); // abnormal return if we get through timeout loop without busy going away
}

NTSTATUS
Port9240DeviceControl(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP pIrp
    )

{
    PIO_STACK_LOCATION  irpSp;
    NTSTATUS            ntStatus = STATUS_SUCCESS;

    ULONG               inBufLength;   /* Input buffer length */
    ULONG               outBufLength;  /* Output buffer length */

    PUCHAR              CharBuffer;
    PUSHORT             ShortBuffer;
    PULONG              LongBuffer;
    PVOID               ioBuffer;
    int                 bytecount, idx, TimeOutCount,TimeOut;
    PUCHAR              lptporta, lptportb, lptportc;  // filled in by WRITE_9240 params
    UCHAR               data;



    irpSp = IoGetCurrentIrpStackLocation( pIrp );
    inBufLength = irpSp->Parameters.DeviceIoControl.InputBufferLength;
    outBufLength = irpSp->Parameters.DeviceIoControl.OutputBufferLength;

    ioBuffer    = pIrp->AssociatedIrp.SystemBuffer;

    CharBuffer  = (PUCHAR) ioBuffer;
    ShortBuffer = (PUSHORT) ioBuffer;
    LongBuffer  = (PULONG) ioBuffer;

    switch ( irpSp->Parameters.DeviceIoControl.IoControlCode )
     {
      // byte 0,1 ShortBuffer[0] port to read
      // returns value read from port in byte 0
      case IOCTL_READ_PORT_UCHAR:

            if ((inBufLength >= 2) && (outBufLength >= 1))
              {
              data = READ_PORT_UCHAR((PUCHAR)ShortBuffer[0]);
              KdPrint( ("Port9240: IOCTL_READ_PORT_UCHAR 0x%X value %X",ShortBuffer[0],data) );
              CharBuffer[0] = data;
              }
            else
              ntStatus = STATUS_BUFFER_TOO_SMALL;
            pIrp->IoStatus.Information = 1; /* Output Buffer Size */
            ntStatus = STATUS_SUCCESS;
            break;

      // byte 0,1 ShortBuffer[0] port to write
      // byte 2   CharBuffer[2]  data to write to port
      case IOCTL_WRITE_PORT_UCHAR:
            if (inBufLength >= 3)
              {
              KdPrint( ("Port9240: IOCTL_WRITE_PORT_UCHAR(0x%X,0x%X)",ShortBuffer[0], CharBuffer[2]) );
              WRITE_PORT_UCHAR((PUCHAR)ShortBuffer[0], CharBuffer[2]);
              }
            else
              ntStatus = STATUS_BUFFER_TOO_SMALL;
            pIrp->IoStatus.Information = 0; /* Output Buffer Size */
            ntStatus = STATUS_SUCCESS;
            break;

      /* byte 0-1  shortbuffer[0] lpt base
         byte 2    charbuffer[2] flag  e.g. 8 for cmd, 2 for data, 0 for wave0, 4 for wave1
         byte 3    charbuffer[3] data to send
         byte 4-7  LongBuffer[1] timeout loop count */
      case IOCTL_WRITE_9240:
            if ((inBufLength >= 7) && (outBufLength >= 1))
              {
              KdPrint( ("Port9240: IOCTL_WRITE_9240(0x%X,0x%X,0x%X)",ShortBuffer[0], CharBuffer[2], CharBuffer[3]) );
              lptporta = (PUCHAR) (ShortBuffer[0] + 2);
              lptportb = (PUCHAR) ShortBuffer[0];
              lptportc = (PUCHAR) (ShortBuffer[0] + 1);
              TimeOutCount = LongBuffer[1];
              TimeOut = WaitNotBusy(TimeOutCount,lptportc);  // wait for 9240 to be ready for next character
              WRITE_PORT_UCHAR(lptportb, CharBuffer[3]);  // data to data port
              WRITE_PORT_UCHAR(lptporta, CharBuffer[2]);  //clock in data with a strobe to control port
              WRITE_PORT_UCHAR(lptporta, CharBuffer[2]+1);
              WRITE_PORT_UCHAR(lptporta, CharBuffer[2]);
              CharBuffer[0] = (UCHAR) TimeOut;
              }
            else
              ntStatus = STATUS_BUFFER_TOO_SMALL;
            pIrp->IoStatus.Information = 1; /* Output Buffer Size */
            ntStatus = STATUS_SUCCESS;
            break;

      /* byte 0-1  shortbuffer[0] lpt base
         byte 2-3  shortbuffer[1] number of bytes to send
         byte 4-7  LongBuffer[1] timeout loop count
         byte 8    charbuffer[8] flag  e.g. 8 for cmd, 2 for data, 0 for wave0, 4 for wave1
         byte 9-n  charbuffer[9]... data to send */
      case IOCTL_WRITE_9240_STRING:
            if ((inBufLength >= 9) && (outBufLength >= 1))
              {
              KdPrint( ("Port9240: IOCTL_WRITE_9240_STRING(0x%X,%d,0x%X)",ShortBuffer[0], ShortBuffer[1], CharBuffer[8]) );
              lptporta = (PUCHAR) (ShortBuffer[0] + 2);
              lptportb = (PUCHAR) ShortBuffer[0];
              lptportc = (PUCHAR) (ShortBuffer[0] + 1);
              bytecount = ShortBuffer[1];
              TimeOutCount = LongBuffer[1];

              for (idx = 0; idx < bytecount; idx++)
                {
                TimeOut = WaitNotBusy(TimeOutCount,lptportc);// wait for 9240 to be ready for next character
                if(TimeOut) break;
                WRITE_PORT_UCHAR(lptportb, CharBuffer[9+idx]);// send data byte
                WRITE_PORT_UCHAR(lptporta, CharBuffer[8]);// create a postive strobe that gets inverted by pc hardware
                WRITE_PORT_UCHAR(lptporta, CharBuffer[8]+1);// to negative strobe
                WRITE_PORT_UCHAR(lptporta, CharBuffer[8]);
                }
              CharBuffer[0] = (UCHAR) TimeOut;
              }
            else
              ntStatus = STATUS_BUFFER_TOO_SMALL;
            pIrp->IoStatus.Information = 1; /* Output Buffer Size */
            ntStatus = STATUS_SUCCESS;
            break;

      default:
            KdPrint( ("Port9240: Unsupported IOCTL Call\n") );
            ntStatus = STATUS_UNSUCCESSFUL;
            pIrp->IoStatus.Information = 0;
            break;

    }
    pIrp->IoStatus.Status = ntStatus;
    IoCompleteRequest( pIrp, IO_NO_INCREMENT );
    return ntStatus;
}

VOID Port9240Unload(IN PDRIVER_OBJECT DriverObject)
{
    WCHAR DOSNameBuffer[] = L"\\DosDevices\\Port9240";
    UNICODE_STRING uniDOSString;

    KdPrint( ("Port9240: Port9240 is Unloading . .\n") );

    RtlInitUnicodeString(&uniDOSString, DOSNameBuffer);
    IoDeleteSymbolicLink (&uniDOSString);
    IoDeleteDevice(DriverObject->DeviceObject);
}
