{  Copyright 2005 Allen Datagraph Systems - All Rights Reserved
   Permission is given to owners of Allen Datagraph 9240 chart recorders to use
   and modify this program for the purpose of talking to the Allen Datagraph
   9240 strip chart recorder.
}

unit Port9240;

interface

uses windows, dialogs, sysutils, winsvc;

const
  OpenRetryCnt         = 5;

  LPT1 = $378;
  LPT2 = $278;

  // data register LPTPORT
  WData    = 0;  // write to data port

  // status register
  RStatus  = 1;  // read status register
    MaskBusy     = $80;  // pin 11     busy status
    MasknAck     = $40;  // pin 10
    MaskPaperOut = $20;  // pin 12
    MaskSelectIn = $10;  // pin 13
    MaskError    = $8;   // pin 15     Paper out status
    MaskInverted = MaskBusy;

    WBusy = MaskBusy;
    WPaperOut = MaskError;

  // control register
  WControl = 2;  // write to control port
    MaskBiDirectional = $20;
    MaskEnableIRQ     = $10;
    MaskSelectPrinter = $8;   // pin 17    CMD/~DATA
          //  this would normally be data/~cmd but because the control output from the pc
          //  inverted its cmd/~data
    MaskInitialize    = $4;   // pin 16    Channel
    MaskLineFeed      = $2;   // pin 14    A/~W
    MaskStrobe        = 1;    // pin 1     strobe
          // this pin is also inverted by the pc.  The port i/o driver sends a
          // positive clock instead of a negative clock to counter act this inversion.
    //WControlInverted  = MaskStrobe + MaskSelectPrinter;


    WCommand = MaskSelectPrinter;
    WDataByte = 0;
    WChannel0 = 0;
    WChannel1 = MaskInitialize;
    WWave     = 0;
    WAlpha    = MaskLineFeed;
    WStrobe   = MaskStrobe;
    WStepMotor = MaskInitialize;

  // 9240 control values
  WriteWave0   = WDatabyte + WChannel0 + WWave;       // = 0 CMD/~DATA = Data,    Channel = 0, W/~A = Wave
  WriteData    = WDatabyte +             WAlpha;      // = 2 CMD/~DATA = DATA,    Channel = X, W/~A = Alpha
  WriteWave1   = WDataByte + WChannel1 + WWave;       // = 4 CMD/~DATA = Data,    Channel = 1, W/~A = Wave
  WriteCommand = WCommand;                            // = 8 CMD/~DATA = Command, Channel = X, W/~A = X
  // we will hook up a sp double throw switch with common hooked up to (maskinitialize) and
  // s1 position to channel (pin 13 of 9240) and s2 position to ~STEP (pin 10 of 9240)
  // we need to position the switch to s1 when mode 80000220 is uncheck and s2 when it is checked

  AllenParPort_TYPE        = cardinal(32700) shl 16; //* 32768-65535 are reserved for customers */
  AllenName                = 'Port9240';
  AllenNameL : widestring  = 'Port9240';
  METHOD_BUFFERED          = 0;
  FILE_ANY_ACCESS          = 0;
//#define CTL_CODE( DeviceType, Function, Method, Access ) (                 \
//    ((DeviceType) << 16) | ((Access) << 14) | ((Function) << 2) | (Method) \
//)  from devioctl.h

  // The IOCTL function codes from 0x800 to 0xFFF are for customer use.
  // read port address
      // byte 0,1 ShortBuffer[0] port to read
  IOCTL_READ_PORT_UCHAR: cardinal = (AllenParPort_TYPE) or
                                    (FILE_ANY_ACCESS shl 14) or
                                    ($904 shl 2) or
                                    METHOD_BUFFERED;

  // write port address
      // byte 0,1 ShortBuffer[0] port to write
      // byte 2   CharBuffer[2]  data to write to port
  IOCTL_WRITE_PORT_UCHAR: cardinal = (AllenParPort_TYPE) or
                                     (FILE_ANY_ACCESS shl 14) or
                                     ($905 shl 2) or
                                     METHOD_BUFFERED;

  // write to 9240 port 1 byte
      (* byte 0-1  shortbuffer[0] lpt base
         byte 2    charbuffer[2] flag  e.g. 8 for cmd, 2 for data, 0 for wave0, 4 for wave1
         byte 3    charbuffer[3] data to send
         byte 4-7  LongBuffer[1] timeout loop count *)
  IOCTL_WRITE_9240: cardinal = (AllenParPort_TYPE) or
                             (FILE_ANY_ACCESS shl 14) or
                             ($906 shl 2) or
                             METHOD_BUFFERED;

  // write to 9240 port multibyte
      (* byte 0-1  shortbuffer[0] lpt base
         byte 2-3  shortbuffer[1] number of bytes to send
         byte 4-7  LongBuffer[1] timeout loop count
         byte 8    charbuffer[8] flag  e.g. 8 for cmd, 2 for data, 0 for wave0, 4 for wave1
         byte 9-n  charbuffer[9]... data to send *)
  IOCTL_WRITE_9240_STRING: cardinal = (AllenParPort_TYPE) or
                             (FILE_ANY_ACCESS shl 14) or
                             ($907 shl 2) or
                             METHOD_BUFFERED;

type



  TFileStatistic = record
    date : _FILETIME; {   dwLowDateTime: DWORD; dwHighDateTime: DWORD;}
    size : LARGE_INTEGER;
  end;

  TPort9240 = class
    private
      FHandle             : THANDLE;         // handle for open/close/read/write/device control
      FLPTPortHandle      : THandle;         // handle to reserve parallel port
      FMsg                : string;          // last error message output
      FTimeout            : integer;         // timeout on get_message before terminate
      FRegBase            : integer;         // register base (lpt1, lpt2)
      FStopSending        : boolean;         // set to true to cause premature return from send loop
      FLastPortNum        : integer;         // records last port num value for step routine

      procedure   ClosePort;                       // close port
      function    GetPaperOut : boolean;
      function    inportb(PortAddress: integer) : integer;
      procedure   outportb(PortAddress, value: integer);
      function    InstallDriver : boolean;
      function    Open : boolean;
      function    StartDriver : boolean;
      function    UninstallDriver : boolean;
      function    UpdateDriver : boolean;          // if latest driver is not installed stop driver and update it
    public
      mode80000220 : boolean;
      constructor Create;                          // class create
      destructor  destroy ; override;              // class destroy
      procedure   init;                            // initialize class
      procedure   setport;                         // update port base from option id
      procedure   stepmotor;                       // steps motor if switch in cable is set to s2 position
      function    Status : integer;                // get status of 9240
      procedure   write(offset,value: integer);    // write(wdata | wcontrol, value)
      function    write9240(portnum, value : integer): boolean;    // write9240( WriteWave0 |WriteData | WriteWave1 | WriteCommand , value);
      function    write9240String(portnum : integer; value : string):boolean; // write9240String( WriteWave0 |WriteData | WriteWave1 | WriteCommand , value);

      property PaperOut           : boolean         read GetPaperOut;
      property PortHandle         : THANDLE         read FHandle;
      property RegBase            : integer         read FRegBase        write FRegBase;
      property StopSending        : boolean         read FStopSending    write FStopSending;
      property Timeout            : Integer         read FTimeout        write FTimeout;
  end;

var
  PortIO : TPort9240;

implementation

uses
  forms,
  option,
  math,
  main;


//////////////////////////////////////////////////////////////////////////////////////

function GetFileStat(FileName : string) : TFileStatistic;
var
  Handle: THandle;
  FindData: TWin32FindData;
begin
  Handle := FindFirstFile(PChar(FileName), FindData);
  if Handle <> INVALID_HANDLE_VALUE then
  begin
    Windows.FindClose(Handle);
    if (FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) = 0 then
    begin
      result.date := FindData.ftLastWriteTime;
      result.size.LowPart := FindData.nFileSizeLow;
      result.size.HighPart := FindData.nFileSizeHigh;
      exit;
    end;
  end;
  Result.date.dwLowDateTime := 0;
  result.date.dwHighDateTime := 0;
  result.size.QuadPart := -1;
end;

procedure TPort9240.ClosePort;
begin
  CloseHandle(FLPTPortHandle);
  CloseHandle(FHandle);
  FLPTPortHandle := INVALID_HANDLE_VALUE;
  FHandle := INVALID_HANDLE_VALUE;
end;

constructor TPort9240.Create;
begin
  inherited create;
  FTimeout := 100000;
  FLPTPortHandle := INVALID_HANDLE_VALUE;
  FHandle := INVALID_HANDLE_VALUE;
end;

destructor TPort9240.destroy;
begin
  if FHandle <> INVALID_HANDLE_VALUE then
    ClosePort;
  inherited destroy;
end;

function TPort9240.GetPaperOut : boolean;
begin
  result := (status and WPaperOut) = 0;
end;

procedure TPort9240.init;
var
  retry : integer;
begin
  if FHandle <> INVALID_HANDLE_VALUE then exit; // init already done
  if comparetext(opt.LPTPort,'lpt1') = 0 then
    FRegBase := LPT1
  else if comparetext(opt.LPTPort,'lpt2') = 0 then
    FRegBase := LPT2
  else
    FRegBase := strtoint(opt.LptPort);
  for retry := 1 to OpenRetryCnt do
    if Open then
      break
    else
    begin
      fmsg := 'init:Open unsuccessful.  retrying';
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
    end;
end;

function  TPort9240.inportb(PortAddress: integer) : integer;
var
  error : boolean;
  BytesReturned : cardinal;
  Buffer : record
        case integer of
        0: (bytes : array[0..2] of byte;);
        1: (value : word;);
  end;
begin
  Buffer.value := PortAddress;
  error := DeviceIoControl(FHandle,
                           IOCTL_READ_PORT_UCHAR,
                           @Buffer.bytes,
                           2,
                           @Buffer.bytes,
                           1,
                           BytesReturned,
                           Nil);
  if not error then
  begin
    FMsg := 'inportb:Error occured while talking to ' + AllenName + ' driver ' + SysErrorMessage(getLastError);
    sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
  end;
  result := integer(buffer.bytes[0]);
end;

function TPort9240.InstallDriver : boolean;
var
  SchSCManager   : SC_HANDLE;
  schService     : SC_HANDLE;
  err            : longword;
  DriverFileName : array[0..79] of char;
begin
  result := false;
  FMsg := 'InstallDriver:Installing ' + AllenName + ' port driver';
  sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
  //* Get Current Directory. Assumes Port9240.SYS driver is in this directory.    */
  //* Doesn't detect if file exists, nor if file is on removable media - if this  */
  //* is the case then when windows next boots, the driver will fail to load and  */
  //* a error entry is made in the event viewer to reflect this */

  //* Get System Directory. This should be something like c:\windows\system32 or  */
  //* c:\winnt\system32 with a Maximum Character lenght of 20. As we have a       */
  //* buffer of 80 bytes and a string of 25 bytes to append, we can go for a max  */
  //* of 54 bytes */

  if GetSystemDirectory(DriverFileName, 54) = 0 then
  begin
    FMsg := 'InstallDriver:Failed to get System Directory. Is System Directory Path > 54 Characters?' + #13 + #10 +
            'Please manually copy driver ' + AllenName + '.sys to your system32/driver directory.';
    sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
    exit;
  end;

  //* Append our Driver Name */
  lstrcat(DriverFileName,'\Drivers\' + AllenName + '.sys');
  FMsg := 'InstallDriver:Copying driver to ' + DriverFileName;
  sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);

  //* Copy Driver to System32/drivers directory. This fails if the file doesn't exist. */

  if not CopyFile(pchar(extractfilepath(application.exename) + AllenName + '.sys'), DriverFileName, FALSE) then
  begin
    FMsg := 'InstallDriver:Failed to copy driver to ' + DriverFileName + #13 + #10 +
            'Please manually copy driver to your system32/driver directory.';
    sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
  end;

  //* Open Handle to Service Control Manager */
  SchSCManager := OpenSCManager (Nil,                   //* machine (NULL == local) */
                                 Nil,                   //* database (NULL == default) */
                                 SC_MANAGER_ALL_ACCESS); //* access required */

  if SchSCManager = 0 then
  begin
    if GetLastError = ERROR_ACCESS_DENIED then
    begin
       //* We do not have enough rights to open the SCM, therefore we must */
       //* be a poor user with only user rights. */
       FMsg := 'InstallDriver:You do not have rights to access the Service Control Manager and' + #13 + #10 +
               'the ' + AllenName + ' driver is not installed or started. Please ask' + #13 + #10 +
               'your administrator to install the driver on your behalf.';
       sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
       exit;
    end;
  end;

  try
    //* Create Service/Driver - This adds the appropriate registry keys in */
    //* HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services - It doesn't  */
    //* care if the driver exists, or if the path is correct.              */

    schService := CreateService (SchSCManager,                      //* SCManager database */
                                 AllenName,                         //* name of service */
                                 'Port 9240 parallel port driver',  //* name to display */
                                 SERVICE_ALL_ACCESS,                //* desired access */
                                 SERVICE_KERNEL_DRIVER,             //* service type */
                                 SERVICE_DEMAND_START,              //* start type SERVICE_SYSTEM_START */
                                 SERVICE_ERROR_NORMAL,              //* error control type */
                                 'System32\Drivers\' + AllenName + '.sys', //* service's binary */
                                 Nil,                              //* no load ordering group */
                                 Nil,                              //* no tag identifier */
                                 Nil,                              //* no dependencies */
                                 Nil,                              //* LocalSystem account */
                                 Nil                               //* no password */
                                 );

    if schService = 0 then
    begin
      err := GetLastError;
      if err = ERROR_SERVICE_EXISTS then
        FMsg := 'InstallDriver:Driver already exists. No action taken.'
      else
        FMsg := 'InstallDriver:Unknown error ' + SysErrorMessage(err) + ' while creating Service.';
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      exit;
    end
    else
    begin
      FMsg := 'InstallDriver:Driver successfully installed.';
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
    end;

    //* Close Handle to Service Control Manager */
    CloseServiceHandle (schService);
    result := true;
  finally
    CloseServiceHandle (SchSCManager);
  end;
end;


function TPort9240.Open : boolean;
var
  lptname : string;
var
  DriverFileName : array[0..79] of char;
  sys,new : TFileStatistic;
begin
  result := false;
  if FHandle = INVALID_HANDLE_VALUE then
  begin
    //* Open  Driver. If we cannot open it, try installing and starting it */
    if FLPTPortHandle = INVALID_HANDLE_VALUE then
    begin
      lptname := 'LPT';
      if FRegBase = LPT1 then
        lptname := lptname + '1'
      else if FRegBase = LPT2 then
        lptname := lptname + '2';
      if lptname <> 'LPT' then
      begin
        FLPTPortHandle := CreateFile(pchar(lptname),
                                     GENERIC_READ or GENERIC_WRITE,
                                     0,    // no sharing
                                     Nil,
                                     OPEN_EXISTING,
                                     FILE_ATTRIBUTE_NORMAL,
                                     0);
        if FLPTPortHandle = INVALID_HANDLE_VALUE then
        begin
          FMsg := 'Open:Reserve parallel port unsuccessful. ' + SysErrorMessage(getLastError);
          sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
          exit;
        end;
      end;
    end;
    FHandle := CreateFile('\\.\' + AllenName,
                                      GENERIC_READ or GENERIC_WRITE,
                                      0,
                                      Nil,
                                      OPEN_EXISTING,
                                      FILE_ATTRIBUTE_NORMAL,
                                      0);

    if FHandle = INVALID_HANDLE_VALUE then
    begin
      FMsg := 'Open:Open driver unsuccessful.  Will try to start driver';
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      // only reason we cannot open driver is because it is not running.  So before we
      // start it, verify that it is the same as in the exe directory.
      // cannot open driver.  Verify driver is up to date and copy it to drivers directory if not.
      if GetSystemDirectory(DriverFileName, 54) = 0 then
      begin
        FMsg := 'Open:Failed to get System Directory in update driver. Is System Directory Path > 54 Characters?' + #13 + #10 +
                'Cannot determine if driver is up to date.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
        exit;
      end;

      //* Append our Driver Name */
      lstrcat(DriverFileName,'\Drivers\' + AllenName + '.sys');
      sys := GetFileStat(DriverFileName);
      new := GetFileStat(extractfilepath(application.exename) + AllenName + '.sys');
      if (sys.date.dwLowDateTime <> new.date.dwLowDateTime) or
         (sys.date.dwHighDateTime <> new.date.dwHighDateTime) or
         (sys.size.QuadPart <> new.size.QuadPart) then
      begin
        if not CopyFile(pchar(extractfilepath(application.exename) + AllenName + '.sys'), DriverFileName, FALSE) then
        begin
          FMsg := 'Open:Failed to copy driver to ' + DriverFileName + #13 + #10 +
                  'Please manually copy driver to your system32/driver directory.';
          sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
        end;
      end;
      //* Start or Install AllenParPort Driver */
      if not StartDriver then
      begin
        FMsg := 'OpenAllenParPort:Start driver unsuccessful.  retry.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
        if not StartDriver then
        begin
          FMsg := 'OpenAllenParPort:Start driver unsuccessful 2 times.  I give up.';
          sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
          exit;
        end;
      end;
      //* Then try to open once more, before failing */
      FHandle := CreateFile('\\.\' + AllenName,
                                        GENERIC_READ,
                                        0,
                                        Nil,
                                        OPEN_EXISTING,
                                        FILE_ATTRIBUTE_NORMAL,
                                        0);

      if FHandle = INVALID_HANDLE_VALUE then
      begin
        FMsg := 'Open:Open ' + AllenName + ': Couldn''t access Driver, Please ensure driver is loaded.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
        CloseHandle(FLPTPortHandle);
        FLPTPortHandle := INVALID_HANDLE_VALUE;
        exit;
      end
      else
      begin
        FMsg := 'Open:Driver opened successfully after starting driver.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      end;
    end
    else
      updatedriver;
  end;
  result := true;
end;

procedure TPort9240.outportb(PortAddress, value: integer);
var
  error : boolean;
  BytesReturned : cardinal;
  Buffer : record
        case integer of
        0: (bytes : array[0..2] of byte;);
        1: (value : word;);
  end;
begin
  Buffer.value := PortAddress;
  Buffer.bytes[2] := value;
  error := DeviceIoControl(FHandle,
                           IOCTL_WRITE_PORT_UCHAR,
                           @Buffer.bytes,
                           3,
                           Nil,
                           0,
                           BytesReturned,
                           Nil);
  if not error then
  begin
    FMsg := 'outportb:Error occured while talking to ' + AllenName + ' driver ' + SysErrorMessage(getLastError);
    sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
  end;
end;

function  TPort9240.StartDriver : boolean;
var
    SchSCManager  : SC_HANDLE;
    schService    : SC_HANDLE;
    ret           : boolean;
    err           : longword;
    str           : pchar;
begin
  result := false;
  fmsg := 'StartDriver:Starting driver';
  sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);

  //* Open Handle to Service Control Manager */
  SchSCManager := OpenSCManager (Nil,                        //* machine (NULL == local) */
                                 Nil,                        //* database (NULL == default) */
                                 SC_MANAGER_ALL_ACCESS);      //* access required */

  if SchSCManager = 0 then
  begin
    if GetLastError = ERROR_ACCESS_DENIED then
    begin
       //* We do not have enough rights to open the SCM, therefore we must */
       //* be a poor user with only user rights. */
       FMsg := 'StartDriver:You do not have rights to access the Service Control Manager and' + #13 + #10 +
               'the ' + AllenName + ' driver is not installed or started. Please ask' + #13 + #10 +
               'your administrator to install the driver on your behalf.';
       sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
       exit;
    end;
  end;

  try
    schService := 0;
    sleep(200);
    while schService = 0 do
    begin
      //* Open a Handle to the  Service Database */
      schService := OpenService(SchSCManager,         //* handle to service control manager database */
                                AllenName,            //* pointer to name of service to start */
                                SERVICE_ALL_ACCESS);  //* type of access to service */

      if schService = 0 then
      begin
        case GetLastError of
          ERROR_ACCESS_DENIED:
            begin
              FMsg := 'StartDriver:You do not have rights to the ' + AllenName + ' service database' + #13 + #10;
              sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
              exit;
            end;
          ERROR_INVALID_NAME:
            begin
              FMsg := 'StartDriver:The specified service name is invalid.' + #13 + #10;
              sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
              exit;
            end;
          ERROR_SERVICE_DOES_NOT_EXIST:
            begin
              FMsg := 'StartDriver:The ' + AllenName + ' driver does not exist.  Installing driver.';
              sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
              if not InstallDriver then
              begin
                fmsg := 'StartDriver:Install failed';
                sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
                exit;
              end
              else
              begin
                fmsg := 'StartDriver:Install successful';
                sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
              end;
              //break;
            end;
        end;
      end;
    end;
    try
      str := nil;
      //* Start the  Driver. Errors will occur here if .SYS file doesn't exist */
      ret := StartService (schService,    //* service identifier */
                           0,             //* number of arguments */
                           str);          //* pointer to arguments */

      if ret then
      begin
        FMsg := 'StartDriver:The ' + AllenName + ' driver has been successfully started.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      end
      else
      begin
          err := GetLastError;
          if err = ERROR_SERVICE_ALREADY_RUNNING then
          begin
            FMsg := 'StartDriver:The ' + AllenName + ' driver is already running.';
            sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
          end
          else
          begin
            FMsg := 'StartDriver:Unknown error (' + SysErrorMessage(err) + ') while starting ' + AllenName + ' driver service.' + #13 + #10 +
                    'Does ' + AllenName + '.SYS exist in your \System32\Drivers Directory?';
            sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
            exit;
          end;
      end;

      result := true;
      //* Close handle to Service Control Manager */
    finally
      CloseServiceHandle (schService);
    end;
  finally
    CloseServiceHandle (SchSCManager);
  end;
end;

procedure TPort9240.setport;           // set port id
begin
  if comparetext(opt.LPTPort,'lpt1') = 0 then
    FRegBase := LPT1
  else if comparetext(opt.LPTPort,'lpt2') = 0 then
    FRegBase := LPT2
  else
    FRegBase := strtoint(opt.LptPort);
end;

function TPort9240.Status : integer;
begin
  if FHandle = INVALID_HANDLE_VALUE then init;
  result := inportb(FRegBase + RStatus);
end;

function TPort9240.UninstallDriver : boolean;
var
  SchSCManager   : SC_HANDLE;
  schService     : SC_HANDLE;
  ret            : boolean;
  DriverFileName : array[0..79] of char;
  servicestatus  : SERVICE_STATUS;
begin
  result := false;
  FMsg := 'UninstallDriver:Uninstalling driver ' + #13 + #10;
  sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
  if FHandle <> INVALID_HANDLE_VALUE then
    ClosePort;
  //* Open Handle to Service Control Manager */
  SchSCManager := OpenSCManager (nil,                    // machine (NULL == local)
                                 nil,                    // database (NULL == default)
                                 SC_MANAGER_ALL_ACCESS); // access required

  if SchSCManager = 0 then
  begin
    if GetLastError = ERROR_ACCESS_DENIED then
    begin
       //* We do not have enough rights to open the SCM, therefore we must */
       //* be a poor user with only user rights. */
       FMsg := 'UninstallDriver:You do not have rights to access the Service Control Manager and' + #13 + #10 +
               'the ' + AllenName + ' driver is not installed or started. Please ask' + #13 + #10 +
               'your administrator to install the driver on your behalf.';
       sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
       exit;
    end;
  end;

  try
    //* Open Handle to  Service Database */
    schService := OpenService (SchSCManager,
                               AllenName ,
                               SERVICE_ALL_ACCESS);

    if schService = 0 then
    begin
      FMsg := 'UninstallDriver:Error while opening ' + AllenName + ' service, has ' + AllenName  + ' been installed?';
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
    end;

    try
      //* Stop Service */
      ret := ControlService (schService,
                             SERVICE_CONTROL_STOP,
                             serviceStatus);

      if ret then
      begin
        FMsg := 'UninstallDriver:' + AllenName + ' service has been successfully stopped.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      end
      else
      begin
        FMsg := 'UninstallDriver:Unknown error (' + SysErrorMessage(getLastError) + ') while stopping ' + AllenName + ' service.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      end;

      //* Delete Service */
      ret := DeleteService (schService);

      if ret then
      begin
        FMsg := 'UninstallDriver:' + AllenName + ' service has been successfully deleted.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      end
      else
      begin
        FMsg := 'UninstallDriver:Error ' + SysErrorMessage(getLastError) + ') removing ' + AllenName + ' service - ' + AllenName + ' has NOT been successfully removed.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      end;


      //* Get Current Directory. Assumes .SYS driver is in this directory.    */
      //* Doesn't detect if file exists, nor if file is on removable media - if this  */
      //* is the case then when windows next boots, the driver will fail to load and  */
      //* a error entry is made in the event viewer to reflect this */

      //* Get System Directory. This should be something like c:\windows\system32 or  */
      //* c:\winnt\system32 with a Maximum Character lenght of 20. As we have a       */
      //* buffer of 80 bytes and a string of 25 bytes to append, we can go for a max  */
      //* of 54 bytes */

      if GetSystemDirectory(DriverFileName, 54) = 0 then
      begin
        FMsg := 'UninstallDriver:Failed to get System Directory. Is System Directory Path > 54 Characters?' + #13 + #10 +
                'Please manually copy driver ' + AllenName + '.sys to your system32/driver directory.';
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
        exit;
      end;

      //* Append our Driver Name */
      lstrcat(DriverFileName,'\Drivers\' + AllenName + '.sys');
      FMsg := 'UninstallDriver:Deleting driver file from ' + DriverFileName + #13 + #10;
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);

      //* Delete Driver from System32/drivers directory. This fails if the file doesn't exist or open. */
      if not DeleteFile(DriverFileName) then
      begin
        FMsg := 'UninstallDriver:Failed to delete driver from ' + DriverFileName + #13 + #10 +
                'Please manually delete driver from your system32/driver directory.' + #13 + #10 +
                SysErrorMessage(getLastError);
        sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
        exit;
      end;
      FMsg := 'UninstallDriver:The driver ' + AllenName + ' can now be reinstalled.' + #13 + #10;
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
      result := true;
    finally
      //* Close Handle to  Service Database */
      CloseServiceHandle (schService);
    end;
  finally
    //* Close Handle to Service Control Manager */
    CloseServiceHandle(SchSCManager);
  end;
end;

function TPort9240.UpdateDriver : boolean;
var
  DriverFileName : array[0..79] of char;
  sys,new : TFileStatistic;
begin
  result := false;
  if GetSystemDirectory(DriverFileName, 54) = 0 then
  begin
    FMsg := 'UpdateDriver:Failed to get System Directory in update driver. Is System Directory Path > 54 Characters?' + #13 + #10 +
            'Cannot determine if driver is up to date.';
    sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
    exit;
  end;

  //* Append our Driver Name */
  lstrcat(DriverFileName,'\Drivers\' + AllenName + '.sys');
  sys := GetFileStat(DriverFileName);
  new := GetFileStat(extractfilepath(application.exename) + AllenName + '.sys');
  if (sys.date.dwLowDateTime <> new.date.dwLowDateTime) or
     (sys.date.dwHighDateTime <> new.date.dwHighDateTime) or
     (sys.size.QuadPart <> new.size.QuadPart) then
  begin
     fmsg := 'UpdateDriver:The driver ' + AllenName + ' is out of date.  Updating';
     sendmessage(application.MainForm.Handle,WM_logmessage,integer(fmsg),1);
     result := UninstallDriver;
     if result then result := Open;
     if result then
     begin
       fmsg := 'UpdateDriver:driver updated';
       sendmessage(application.MainForm.Handle,WM_logmessage,integer(fmsg),1);
     end;
  end
  else
     result := true;
end;

procedure TPort9240.write(offset,value: integer);
begin
  if FHandle = INVALID_HANDLE_VALUE then init;
  outportb(FRegBase+offset,value);
end;

procedure   TPort9240.stepmotor;                       // steps motor if switch in cable is set to s2 position
begin
  write(wcontrol,FLastPortNum - WStepMotor);
  mainform.delay(0.000050);  // delay 50 us.
  write(wcontrol,FLastPortNum);
end;



// write9240( WriteWave0 |WriteData | WriteWave1 | WriteCommand , value);
// write a value to the parallel port and strobes it to the 9240
// returns false if no timeout occurred and data was written to 9240
// returns true if timeout occurred.  No data written to port
// a timeout occurs if the busy signal does not go away withing the timeout period
function TPort9240.Write9240( portnum, value : integer ) : boolean;
var
  error : boolean;
  BytesReturned : cardinal;
  Buffer : record
        case integer of
        0: (bytes : array[0..8] of byte;);
        1: (value : array[0..1] of smallint;);
        2: (longs : array[0..1] of integer;);
  end;
begin
  if FHandle = INVALID_HANDLE_VALUE then init;
      (* byte 0-1  shortbuffer[0] lpt base
         byte 2    charbuffer[2] flag  e.g. 8 for cmd, 2 for data, 0 for wave0, 4 for wave1
         byte 3    charbuffer[3] data to send
         byte 4-7  LongBuffer[1] timeout loop count *)
  Buffer.value[0] := FRegBase;
  if mode80000220 then portnum := portnum or WStepMotor;
  Buffer.bytes[2] := portnum;
  FLastPortNum := portnum;
  Buffer.bytes[3] := value;
  Buffer.longs[1] := FTimeout;
  error := DeviceIoControl(FHandle,
                            IOCTL_Write_9240,
                            @buffer.bytes,
                            9,
                            @buffer.bytes,
                            4,
                            BytesReturned,
                            nil);

   if not error then
   begin
     FMsg := 'Write9240:Error occured while talking to ' + AllenName + ' driver ' + SysErrorMessage(getLastError);
     sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
   end;
   result := boolean(buffer.bytes[0]);
end;

// write9240( WriteWave0 |WriteData | WriteWave1 | WriteCommand , value);
// write a list of values to the parallel port and strobe them to the 9240
// returns false if no timeout occurred and data was written to 9240
// returns true if timeout occurred.  No data written to port
// a timeout occurs if the busy signal does not go away withing the timeout period
function TPort9240.write9240String(portnum : integer; value : string):boolean; // write9240String( WriteWave0 |WriteData | WriteWave1 | WriteCommand , value);
const
  maxbytes = 250;  // max bytes per call
var
  error : boolean;
  BytesReturned : cardinal;
  Buffer : record
        case integer of
        0: (bytes : array[0..maxbytes + 9] of byte;);
        1: (value : array[0..1] of smallint;);
        2: (longs : array[0..1] of integer;);
  end;
  beginbyte : integer;
  thislen : integer;
  idx : integer;
begin
  if FHandle = INVALID_HANDLE_VALUE then init;
  if mode80000220 then portnum := portnum or WStepMotor;
  FLastPortNum := portnum;
      (* byte 0-1  shortbuffer[0] lpt base
         byte 2-3  shortbuffer[1] number of bytes to send
         byte 4-7  LongBuffer[1] timeout loop count
         byte 8    charbuffer[8] flag  e.g. 8 for cmd, 2 for data, 0 for wave0, 4 for wave1
         byte 9-n  charbuffer[9]... data to send *)
  result := false;
  FStopSending := false;
  beginbyte := 1;
  while not result and not FStopSending do
  begin
    Buffer.value[0] := FRegBase;
    thislen := length(value) - (beginbyte - 1);
    if thislen > maxbytes then thislen := maxbytes;
    if thislen <= 0 then break;
    Buffer.value[1] := thislen;
    Buffer.longs[1] := FTimeout;
    Buffer.bytes[8] := portnum;
    for idx := 0 to thislen do
      buffer.bytes[idx + 9] := ord(value[beginbyte + idx]);
    error := DeviceIoControl(FHandle,
                              IOCTL_WRITE_9240_STRING,
                              @buffer.bytes,
                              thislen + 9,
                              @buffer.bytes,
                              1,
                              BytesReturned,
                              nil);

    if not error then
    begin
      FMsg := 'write9240string:Error occured while talking to ' + AllenName + ' driver ' + SysErrorMessage(getLastError);
      sendmessage(application.mainform.Handle,WM_Logmessage,integer(fmsg),1);
    end;
    result := boolean(buffer.bytes[0]);
    beginbyte := beginbyte + thislen;
  end;
end;

initialization
  PortIO := TPort9240.Create;
  //ParPort.Zaxis.zvalue

end.
