{  Copyright 2005 Allen Datagraph Systems - All Rights Reserved
   Permission is given to owners of Allen Datagraph 9240 chart recorders to use
   and modify this program for the purpose of talking to the Allen Datagraph
   9240 strip chart recorder.
}
unit main;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, option, Menus, ComCtrls;

const
  delaytime = 0.0015;  // delay between steps
  // the stepper motor cannot step faster than this.

type
  TMainForm = class(TForm)
    ChannelCB: TCheckBox;
    Chan0OnCB: TCheckBox;
    Chan1OnCB: TCheckBox;
    ChartSpeedCB: TComboBox;
    CirclesSquaresButton: TButton;
    DualAnalogButton: TButton;
    exit1: TMenuItem;
    File1: TMenuItem;
    FilterPortCB: TCheckBox;
    FilterPortCoB: TComboBox;
    GraphicsTextButton: TButton;
    HorizontalTextButton: TButton;
    Invert0CB: TCheckBox;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    LPT11: TMenuItem;
    LPT21: TMenuItem;
    MainMenu1: TMainMenu;
    Memo: TMemo;
    Other1: TMenuItem;
    Panel1: TPanel;
    Panel2: TPanel;
    PhotoButton: TButton;
    port1: TMenuItem;
    SelfTestButton: TButton;
    SendTestButton: TButton;
    StopButton: TButton;
    Timer1: TTimer;
    VerticalTextButton: TButton;
    WaveformStartButton: TButton;
    WorstVTextButton: TButton;
    PrintHeadAlignButton: TButton;
    Mode80000220CB: TCheckBox;
    procedure Chan0OnCBClick(Sender: TObject);
    procedure Chan1OnCBClick(Sender: TObject);
    procedure ChannelCBClick(Sender: TObject);
    procedure ChartSpeedCBChange(Sender: TObject);
    procedure DualAnalogButtonClick(Sender: TObject);
    procedure exit1Click(Sender: TObject);
    procedure FilterPortCBClick(Sender: TObject);
    procedure FilterPortCoBChange(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure GraphicsTextButtonClick(Sender: TObject);
    procedure HorizontalTextButtonClick(Sender: TObject);
    procedure Invert0CBClick(Sender: TObject);
    procedure LPT11Click(Sender: TObject);
    procedure LPT21Click(Sender: TObject);
    procedure Other1Click(Sender: TObject);
    procedure PhotoButtonClick(Sender: TObject);
    procedure SelfTestButtonClick(Sender: TObject);
    procedure SendTestButtonClick(Sender: TObject);
    procedure StopButtonClick(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure VerticalTextButtonClick(Sender: TObject);
    procedure WaveformStartButtonClick(Sender: TObject);
    procedure WorstVTextButtonClick(Sender: TObject);
    procedure PrintHeadAlignButtonClick(Sender: TObject);
    procedure Mode80000220CBClick(Sender: TObject);
  private
    { Private declarations }
    CPUSpeed : Int64;
    fquit : boolean;
    FStopSendTest : boolean;
    active : boolean;
    function  PaperOut : boolean;
    procedure WMLogMessage(var Message: Tmessage); message WM_LogMessage;
  public
    { Public declarations }
    procedure delay(length : double);
    procedure sendstring(offset: integer; str : string);
    procedure sendvalue(offset, value : integer);
    procedure stepmotor;
  end;

function  ReadCycleTime : int64;
function  CycleFrequency : int64;

var
  MainForm: TMainForm;

implementation

{$R *.dfm}

uses math, port9240;

procedure TMainForm.Chan0OnCBClick(Sender: TObject);
begin
  if chan0oncb.Checked then
    sendvalue(WriteCommand,$48)
  else
    sendvalue(WriteCommand,$40);
end;

procedure TMainForm.Chan1OnCBClick(Sender: TObject);
begin
  if chan1oncb.Checked then
    sendvalue(WriteCommand,$49)
  else
    sendvalue(WriteCommand,$41);
end;

procedure TMainForm.ChannelCBClick(Sender: TObject);
begin
  if channelcb.Checked then
    sendvalue(WriteCommand,$f3)
  else
    sendvalue(WriteCommand,$f2);
end;

// chart speeds can be changed on the fly during
// waveform test and analog waveform test
procedure TMainForm.ChartSpeedCBChange(Sender: TObject);
const
  itemindextospeedcmd : array[0..8] of byte = ($ec, $eb, $ea, $e9, $f0, $ef, $ee, $ed, $ff);
begin
  memo.Lines.Add('change motor speed to ' + chartspeedcb.Items[chartspeedcb.itemIndex]);
  fquit := false;
  sendvalue(WriteCommand,itemindextospeedcmd[chartspeedcb.ItemIndex]);
end;

// get performance counter frequency by timing a 0.5 second sleep.
function  CycleFrequency : int64;
var
  starttime, endtime : int64;
begin
  starttime := ReadCycleTime;
  sleep(500);
  endtime := ReadCycleTime;
  Result := (endtime - starttime) * 2;
  //QueryPerformanceFrequency(result);
end;

// routine to wait for stepper motor
// length is time delay length in seconds
procedure TmainForm.delay(length : double);
var
  endtime : int64;
begin
  endtime := trunc(readcycletime + length * CPUSpeed);
  while readcycletime < endtime do
    application.ProcessMessages;
end;

// setup to receive two analog waveforms
procedure TMainForm.DualAnalogButtonClick(Sender: TObject);
const
  cmdlist : array[0..9] of byte = ($f9, // reset
                                   $e7, // analog waveform mode
                                   $f2, // dual 2x20mm format
                                   $48, // Channel 0 on
                                   $49, // Channel 1 on
                                   $d0, // chart speed on
                                   $70, // channel 0 5mm grid
                                   $79, // channel 1 5mm grid
                                   $ec, // chart speed 0
                                   $0); // annotation @ bottom
var
  idx : integer;
begin
  memo.Lines.Add('Waveform Test starting');
  fquit := false;
  for idx := 0 to high(cmdlist) do
  begin
    sendvalue(WriteCommand,cmdlist[idx]);
    if idx = 1 then sleep(10);
  end;
  sendstring(WriteData,'Bottom Margin' + #13);
  sendvalue(WriteCommand,15);
  sendstring(WriteData,'Text @ 15mm' + #13);
  FilterPortCBClick(nil);
  while not fquit and not paperout do
  begin
    sleep(10);
    Application.ProcessMessages;
  end;
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

// process file exit
procedure TMainForm.exit1Click(Sender: TObject);
begin
  close;
end;

procedure TMainForm.FilterPortCBClick(Sender: TObject);
begin
  FilterPortCobChange(nil);
end;

procedure TMainForm.FilterPortCoBChange(Sender: TObject);
var
  value : integer;
begin
  if FilterPortCoB.Text = '' then exit;
  value := strtoint(FilterPortCoB.text);
  if filterPortCB.Checked then value := value + $80;  // set filter data from parallel port
  sendvalue(WriteCommand,$e6);
  sendvalue(WriteData,value);
end;

// save lpt setting
procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  opt.save;
  StopButtonClick(nil);
end;

// power up form init
procedure TMainForm.FormCreate(Sender: TObject);
begin
  FStopSendTest := true;
  active := false;
  CPUSpeed := cyclefrequency;
  SetPriorityClass(GetCurrentProcess(),REALTIME_PRIORITY_CLASS);
end;

procedure TMainForm.GraphicsTextButtonClick(Sender: TObject);
const
  // this is an diagonal arrow
  arrowdata : array[0..15] of string = (
    #000 + #248 + #000 + #000,   //  0
    #000 + #192 + #000 + #000,   //  1
    #000 + #160 + #000 + #000,   //  2
    #000 + #144 + #000 + #000,   //  3
    #000 + #136 + #000 + #000,   //  4
    #000 + #004 + #000 + #000,   //  5
    #000 + #002 + #000 + #000,   //  6
    #000 + #001 + #000 + #000,   //  7
    #128 + #000 + #000 + #000,   //  8
    #064 + #000 + #000 + #000,   //  9
    #032 + #000 + #000 + #000,   // 10
    #016 + #000 + #000 + #000,   // 11
    #008 + #000 + #000 + #000,   // 12
    #004 + #000 + #000 + #000,   // 13
    #002 + #000 + #000 + #000,   // 14
    #001 + #000 + #000 + #000);  // 15
var
  row : integer;
  arrow : integer;
begin
  memo.Lines.Add('Graphics Test starting');
  fquit := false;

  sendvalue(WriteCommand,$f9);  // reset
  if not mode80000220cb.Checked then
    sendvalue(WriteCommand,$e2);  // graphics mode
  sleep(10);
  while not fquit and not PaperOut do
  begin
    application.processmessages;
    for row := 0 to 15 do
    begin
      // send a row of arrows 12 arrows
      for arrow := 1 to 2 do
        sendstring(WriteData,arrowdata[row] + arrowdata[row] + arrowdata[row] + arrowdata[row] + arrowdata[row] + arrowdata[row]);
      stepmotor;
      stepmotor;
    end;
  end;
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

procedure TMainForm.HorizontalTextButtonClick(Sender: TObject);
  // routine to create columns of text
  procedure HorizontalArray(fontname : string);
  var
    row, idx, col, ch : integer;
    longest : integer;
    htext : array[1..16] of string;  // 16 rows of text
  begin
    for row := 1 to 16 do
      htext[row] := '';
    htext[3] := fontname; // put font name in 3rd row
    for idx := 0 to 7 do
    begin
      row := 5 + idx;
      for col := 0 to 15 do
      begin
        ch := 16 * idx + col;
        if (ch = 10) or (ch = 13) then ch := 32;
        htext[row] := htext[row] + chr(ch);  // fill up rows with characters from alphabet
      end;
    end;
    // now transpose the matrix to send it to 9240
    longest := 0;
    for row := 1 to 16 do
      if length(htext[row]) > longest then longest := length(htext[row]);
    for col := 1 to longest do
      for row := 16 downto 1 do
      begin
        if length(htext[row]) > col then
          ch := ord(htext[row][col])
        else
          ch := 32;
        sendvalue(WriteData,ch);
        if fquit or paperout then break;
      end;
  end;
begin
  memo.Lines.Add('Horizontal Test starting');
  fquit := false;
  sendvalue(WriteCommand,$f9);  // reset
  sendvalue(WriteCommand,$d8);  // 1mm horizontal
  sleep(10);
  horizontalArray('1mm Horiz. font');
  sendvalue(WriteCommand,$d7);  // 2mm horizontal
  application.processmessages;
  horizontalArray('2mm Horiz. font');
  sendvalue(WriteCommand,$f4);  // 3mm horizontal
  application.processmessages;
  horizontalArray('3mm Horiz. font');
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

procedure TMainForm.Invert0CBClick(Sender: TObject);
begin
  if active then exit;
  active := true;
  try
    if Invert0CB.Checked then
      sendvalue(WriteCommand,$e4)
    else
      sendvalue(WriteCommand,$e5);
  finally
    active := false;
  end;
end;

// process file -> port -> LPT1
procedure TMainForm.LPT11Click(Sender: TObject);
begin
  opt.LPTPort := 'LPT1';
  portio.setport;
end;

// process file -> port -> LPT2
procedure TMainForm.LPT21Click(Sender: TObject);
begin
  opt.LPTPort := 'LPT2';
  portio.setport;
end;

// process file -> port -> other
procedure TMainForm.Other1Click(Sender: TObject);
begin
  // get port resource pointer.  default to lpt1
  opt.LPTPort := InputBox('port base','Enter Port base address.  Preceed hexadecimal numbers with $','$378');
  portio.setport;
end;

function TmainForm.PaperOut : boolean;
begin
  result := portio.PaperOut;
  if result then memo.Lines.Add('Out of paper');
  // memo.lines.Add('status = ' + inttohex(portio.Status,2));
end;

procedure TMainForm.stepmotor;
var
  timecount : integer;
begin
    if Mode80000220CB.Checked then
    begin
      timecount := 0;
      while (portio.Status and MaskBusy) = 0 do
      begin
        inc(timecount);
        if timecount > portio.Timeout then
        begin
          memo.Lines.Add('timeout on last byte complete in step motor');
          break;
        end;
      end;
      portio.stepmotor;
      delay(delaytime);
    end
    else
    begin
      sendvalue(WriteCommand,$f5);
      delay(delaytime);
    end;
end;

// print a photo
procedure TMainForm.PhotoButtonClick(Sender: TObject);
var
  image : TImage;
  row   : integer;
  rowdata : string;
  idx  : integer;
  cols : integer;
  colptr : pchar;
  procedure sendrow;
  begin
    sendstring(WriteData,rowdata);
    // double step
    stepmotor;
    stepmotor;
  end;
  procedure sendblanks(cnt : integer);
  var
    idx : integer;
  begin
    for idx := 1 to length(rowdata) do
      rowdata[idx] := #0;
    for idx := 1 to cnt do
      sendrow;
  end;
begin
  image := TImage.Create(nil);
  // this image was created by using a digital camera
  // then converting to gray scale using photoshop
  // then converting to a black/white image using photograv
  if Sender = PhotoButton then
    image.Picture.LoadFromFile(extractfilepath(application.exename) + '\self photo grave.bmp')
  else
  begin
    image.Picture.Bitmap.PixelFormat := pf1bit;
    image.picture.bitmap.Monochrome := true;
    image.Picture.Bitmap.Width  := 384;
    // to correct for different aspect ratio of stepper (9 dot/mm) vs. horiz resolution of (8 dots/mm) we increase y size by ratio
    image.Picture.Bitmap.Height := 384 * 9 div 8;
    // we draw inverse video since white pixels appear black
    image.Canvas.Brush.Color := clBlack;
    image.Canvas.FillRect(rect(0,0,383,image.Picture.Bitmap.Height));
    // now put some stuff on picture
    image.Canvas.Pen.Color := clWhite;
    image.Canvas.Brush.Color := clWhite;
    image.Canvas.Pen.Width := 1;
    image.Canvas.MoveTo(0,0);         // diagonal cross
    image.Canvas.LineTo(383,image.Picture.Bitmap.Height);
    image.Canvas.MoveTo(383,0);
    image.Canvas.LineTo(0,image.Picture.Bitmap.Height);
    image.Canvas.FrameRect(rect(142,142 * 9 div 8,242,242 * 9 div 8)); // rectangle appears as square
    image.Canvas.Arc(51,51 * 9 div 8,332,332 * 9 div 8,0,0,0,0); // elipse appears as circle
  end;
  fquit := false;
  sendvalue(WriteCommand,$f9);
  if not Mode80000220CB.Checked then
    sendvalue(WriteCommand,$e2);
  sleep(10);
  setlength(rowdata,384 div 8); // a row of pixels
  sendblanks(10);
  // we go backward through image to make it appear right side up on unit
  for row := image.Picture.Bitmap.Height - 1 downto 0 do
  begin
    application.processmessages;
    if fquit or PaperOut then break;
    cols := image.Picture.Bitmap.Width;
    if cols > 384 then cols := 384;
    // get pointer to scanline from bitmap
    colptr := pchar(image.Picture.Bitmap.scanline[row]);
    // put them in image from left to right
    for idx := 1 to (cols + 7) div 8  do
      rowdata[length(rowdata) - idx + 1] := pchar(colptr + idx - 1)^;
    // pad with zeros if image not wide enough
    for idx := (cols + 7) div 8 + 1 to 384 div 8 do
      rowdata[length(rowdata) - idx + 1] := #0;
    sendrow;
  end;
  sendblanks(20);
  image.Free;
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

// fast queryperformancecounter
function ReadCycleTime : int64;
var
  timer : record
    case integer of
      0: (cycles_low : cardinal;
          cycles_high: cardinal);
      1: (cycles : int64);
  end;
begin
  asm
        rdtsc                  // this instruction requires pentium based machine
        mov timer.cycles_high, edx
        mov timer.cycles_low, eax
  end;
  result := timer.cycles;
end;

// if parallel port is getting timeouts this test will
// continuously send stop chart commands so you can
// monitor data on an oscilloscope.  It is probably not
// useful to customers.
procedure TMainForm.SendTestButtonClick(Sender: TObject);
var
  idx : integer;
begin
  memo.Lines.Add('Send stop command until stopped');
  FStopSendTest := false;
  FQuit := false;
  while not FStopSendTest do
  begin
    for idx := 1 to 20 do
    begin
      sendvalue(WriteCommand,$ff);
      fquit := false;
    end;
    application.ProcessMessages;
  end;
  memo.Lines.Add('Send stop complete');
end;

// start selftest
procedure TMainForm.SelfTestButtonClick(Sender: TObject);
begin
  memo.Lines.Add('Self Test starting');
  sendvalue(WriteCommand,$fa);
end;

// send a string to 9240 driver
procedure TmainForm.sendstring(offset: integer; str : string);
begin
  if not fquit then
    if portio.write9240String(offset,str) then
    begin
      fquit := true;
      memo.Lines.Add('timeout occurred sendstring');
    end;
end;

// send a value to 9240 driver
procedure TMainForm.sendvalue(offset, value : integer);
begin
  if offset = WriteCommand then
  begin
    case value of
      $40: chan0oncb.Checked := false;
      $41: chan1oncb.Checked := false;
      $48: chan0oncb.Checked := true;
      $49: chan1oncb.Checked := true;
      $e4: Invert0CB.Checked := true;
      $e5: Invert0CB.Checked := false;
      $e9: ChartSpeedCb.ItemIndex := 3;
      $ea: ChartSpeedCb.ItemIndex := 2;
      $eb: ChartSpeedCb.ItemIndex := 1;
      $ec: ChartSpeedCb.ItemIndex := 0;
      $ed: ChartSpeedCb.ItemIndex := 7;
      $ee: ChartSpeedCb.ItemIndex := 6;
      $ef: ChartSpeedCb.ItemIndex := 5;
      $f0: ChartSpeedCb.ItemIndex := 4;
      $f2: channelcb.Checked := false;
      $f3: channelcb.Checked := true;
      $ff: ChartSpeedCb.ItemIndex := 8;
    end;
  end;
  if not fquit then
    if portio.write9240(offset,value) then
    begin
      fquit := true;
      if FStopSendTest then
        memo.Lines.Add('timeout occurred sendvalue');
    end;
end;

// reset 9240 and send stop chart command
procedure TMainForm.StopButtonClick(Sender: TObject);
begin
  memo.Lines.Add('Stop Test');
  portio.StopSending := true;
  sendvalue(WriteCommand,$f9);
  //portio.write(Wdata,$ff);
  //portio.write(Wcontrol,$ff);
  //sendvalue(WriteCommand,$f9);
  timer1timer(nil);   // send stop paper
  paperout;
  FQuit := true;   // set flag to tell commands to stop
  FStopSendTest := true; // set flag to stop send test
end;

// stop chart on power up
procedure TMainForm.Timer1Timer(Sender: TObject);
begin
  timer1.Enabled := false;
  chartspeedcb.ItemIndex := 8;
  chartspeedcbchange(nil);
end;

procedure TMainForm.VerticalTextButtonClick(Sender: TObject);
var
  charset : string;
  idx : integer;
begin
  memo.Lines.Add('Vertical Test starting');
  fquit := false;
  // create character string to send
  for idx := 0 to 127 do
    if (idx = 10) or (idx = 13) then // no carriage returns or line feeds
      charset := charset + ' '
    else
      charset := charset + chr(idx);
  charset := charset + #13 + #13 + #13;
  sendvalue(WriteCommand,$f9);  // reset
  sendvalue(WriteCommand,$f8);  // send 1mm vertical text
  sleep(10);
  sendstring(WriteData,'1mm vertical font' + #13 + #13);
  sendstring(WriteData,charset);
  sendvalue(WriteCommand,$f7);  // send 2mm vertical text
  sendstring(WriteData,'2mm vertical font' + #13 + #13);
  sendstring(WriteData, charset);
  sendvalue(WriteCommand,$f6); // send 3mm vertical text
  sendstring(WriteData,'3mm vertical font' + #13 + #13);
  sendstring(WriteData,charset);
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

procedure TMainForm.WaveformStartButtonClick(Sender: TObject);
const
  cmdlist : array[0..9] of byte = ($f9, // reset
                                   $e8, // digital waveform mode
                                   $f2, // dual channel format
                                   $48, // Channel 0 on
                                   $49, // Channel 1 on
                                   $70, // channel 0 grid 5mm/1mm
                                   $79, // channel 1 grid 5mm horizontal
                                   $ec, // set chart speed 0
                                   $d0, // print speed on
                                   $d2);// print speed interval = 15cm
  //delaytime = 10000;
var
  freq, deg , omega : double;
  idx : integer;
  w0,w1 : integer;
begin
  memo.Lines.Add('Waveform Test starting');
  fquit := false;
  // send startup commands
  for idx := 0 to high(cmdlist) do
  begin
    sendvalue(WriteCommand,cmdlist[idx]);
    if idx = 1 then sleep(10); // wait a bit to allow power supply to turn on
  end;
  sendvalue(WriteData,15);
  if FilterPortCb.Checked then FilterPortCobChange(nil);
  deg := 0;     // angle for sin wave
  freq := 2;    // frequency of sin wave
  while not fquit and not paperout do
  begin
    omega := (freq * deg * pi) / 180;  // convert to radians
    w0 := trunc(100 * sin(omega) + 128); // calculate waveform value
    if deg < 180 then
      w1 := 64
    else
      w1 := 192;
    deg := deg + 1;
    if deg > 359 then deg := 0;
    //memo.lines.add(inttostr(w0) + ' ' + inttostr(w1));
    sendvalue(WriteWave0,w0); // send waveform value
    sendvalue(WriteWave1,w1);
    delay(delaytime*3); // wait 3 motor step times
  end;
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

// accept a message from port9240 and place it in status messages memo
procedure TMainForm.WMLogMessage(var Message: TMessage);
begin
  if message.WParam <> 0 then
    Memo.Lines.Add(string(message.WParam));
end;

// send line of worst case text to test power supply
procedure TMainForm.WorstVTextButtonClick(Sender: TObject);
var
  charset : string;
  idx : integer;
begin
  memo.Lines.Add('Worst Case Vertical line Test starting');
  fquit := false;
  for idx := ord('0') to ord('_') do
    charset := charset + chr(idx);
  sendvalue(WriteCommand,$f9);
  sendvalue(WriteCommand,$f8);
  sleep(10);
  sendstring(WriteData,'1mm vertical font' + #13 + #13);
  while not fquit and not PaperOut do
  begin
    application.ProcessMessages;
    sendstring(WriteData,charset);
  end;
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;


procedure TMainForm.PrintHeadAlignButtonClick(Sender: TObject);
var
  row : integer;
  arrow : integer;
begin
  memo.Lines.Add('Graphics Test starting');
  fquit := false;

  sendvalue(WriteCommand,$f9);  // reset
  if not mode80000220cb.Checked then
    sendvalue(WriteCommand,$e2);  // graphics mode
  sleep(10);
  while not fquit and not PaperOut do
  begin
    application.processmessages;
    for row := 0 to 300 do
    begin
      sendvalue(WriteData,$ff);
      for arrow := 1 to (384 div 8) - 2 do
        sendvalue(WriteData,0);
      sendvalue(WriteData,$ff);
      stepmotor;
      stepmotor;
    end;
  end;
  sendvalue(WriteCommand,$f9);
  timer1timer(nil);
end;

procedure TMainForm.Mode80000220CBClick(Sender: TObject);
begin
  if Mode80000220cb.Checked then
  begin
    DualAnalogButton.Enabled := false;
    HorizontalTextButton.Enabled := false;
    VerticalTextButton.Enabled := false;
    Waveformstartbutton.Enabled := false;
    chartspeedcb.Enabled := false;
    FilterPortCOb.Enabled := false;
    filterportcb.Enabled := false;
    channelcb.Enabled := false;
    chan0oncb.Enabled := false;
    chan1oncb.Enabled := false;
    invert0cb.Enabled := false;
    if assigned(portio) then
      portio.mode80000220 := true;
    showmessage('switch test cable to 80000220 mode');
  end
  else
  begin
    DualAnalogButton.Enabled := true;
    HorizontalTextButton.Enabled := true;
    VerticalTextButton.Enabled := true;
    Waveformstartbutton.Enabled := true;
    chartspeedcb.Enabled := true;
    FilterPortCOb.Enabled := true;
    filterportcb.Enabled := true;
    channelcb.Enabled := true;
    chan0oncb.Enabled := true;
    chan1oncb.Enabled := true;
    invert0cb.Enabled := true;
    if assigned(portio) then
      portio.mode80000220 := false;
    showmessage('switch test cable to normal mode');
  end;
  stopbuttonclick(nil);
end;

end.

